"use client";

import { Star, ThumbsUp, ThumbsDown, Shield } from "lucide-react";
import type { Product, Category } from "@/types";
import Image from "next/image";
import {
  getAISafetyScore,
  getCleanContentScore,
  getCertificateScore,
  getScoreBadgeColor,
} from "@/lib/score-calculator";
import { Badge } from "@/components/ui/badge";
import { useState } from "react";
import { Card, CardContent } from "@/components/ui/card";
import { Label } from "@/components/ui/label";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { useRouter } from "next/navigation";

interface ProductDetailClientProps {
  product: Product;
  categories?: Category[];
}

function getName(val: any) {
  if (!val) return "";
  if (typeof val === "string") return val;
  if (typeof val === "object" && val !== null && typeof val.name === "string")
    return val.name;
  return String(val);
}

export default function ProductDetailClient({
  product,
  categories = [],
}: ProductDetailClientProps) {
  const router = useRouter();

  const aiScore = getAISafetyScore(product);
  const cleanScore = getCleanContentScore(product);
  const certScore = getCertificateScore(product);
  const totalVotes = product.likes + product.dislikes;
  const popularityScore =
    totalVotes === 0 ? 0 : Math.round((product.likes / totalVotes) * 100);

  const avg = (aiScore + cleanScore + certScore) / 3;
  const computedTotalSafetyScore = Math.round(avg * 0.75 + popularityScore * 0.25);

  const [selectedRating, setSelectedRating] = useState<string>("all");
  const [selectedDateFilter, setSelectedDateFilter] = useState<string>("all");

  const filteredReviews = (product.reviews || [])
    .filter((review) => {
      const matchesRating =
        selectedRating === "all" || review.rating === parseInt(selectedRating);
      const reviewDate = new Date(review.date);
      const now = new Date();
      let matchesDate = true;
      if (selectedDateFilter === "last_month") {
        matchesDate =
          reviewDate >= new Date(now.getFullYear(), now.getMonth() - 1, now.getDate());
      } else if (selectedDateFilter === "last_6_months") {
        matchesDate =
          reviewDate >= new Date(now.getFullYear(), now.getMonth() - 6, now.getDate());
      } else if (selectedDateFilter === "last_year") {
        matchesDate = reviewDate >= new Date(now.getFullYear() - 1, now.getDate());
      }
      return matchesRating && matchesDate;
    })
    .sort((a, b) => new Date(b.date).getTime() - new Date(a.date).getTime());

  const getScoreGradient = (score: number) => {
    if (score >= 80) return "from-green-400 to-green-600";
    if (score >= 60) return "from-yellow-400 to-yellow-600";
    return "from-red-400 to-red-600";
  };

  const categoryName =
    categories.find((cat) => cat.id === product.category)?.name ||
    getName(product.category);
  const icerikList = Array.isArray(product.icerik)
    ? product.icerik
    : typeof product.icerik === "string"
    ? [product.icerik]
    : [];
  const icerikBilgisi =
    typeof product.icerik_bilgisi === "string" ? product.icerik_bilgisi : "";
  const usage = product.usage || "";
  const icerikBilgisiLines = icerikBilgisi.split("\n").filter(Boolean);
  const usageLines = usage.split("\n").filter(Boolean);

  const handleBrandClick = (brandName: string) => {
    router.push(`/?brand=${encodeURIComponent(brandName)}`);
  };

  return (
    <div className="bg-white p-3 sm:p-4 md:p-6 rounded-lg shadow-md overflow-x-hidden">
      <div className="flex flex-col md:flex-row md:gap-x-8 lg:gap-x-12">
        <div className="w-full md:w-1/3 flex flex-col items-center flex-shrink-0">
          <div className="bg-gray-100 rounded-lg flex items-center justify-center w-full max-w-[150px] md:max-w-[200px] mx-auto aspect-square relative">
            <Image
              src={product.image || "/placeholder.svg"}
              alt={product.name}
              fill
              className="object-contain p-2"
            />
          </div>
          <div className="w-full text-center mt-3">
            <h1 className="font-bold text-base sm:text-lg lg:text-xl leading-tight">{product.name}</h1>
            <div
              className="text-sm sm:text-base text-gray-600 mt-1 cursor-pointer hover:text-blue-600 hover:underline"
              onClick={() => handleBrandClick(getName(product.brand))}
            >
              {getName(product.brand)}
            </div>
            <div className="text-xs sm:text-sm text-gray-500 mt-1">
              {categoryName}
              {product.subCategory ? ` / ${getName(product.subCategory)}` : ""}
            </div>
          </div>
        </div>

        <div className="w-full md:w-2/3 mt-6 md:mt-0 md:pl-6">
          <h2 className="font-semibold text-md sm:text-lg text-blue-700 mb-3">Değerlendirme Özeti</h2>
          <div className="flex items-center gap-2 mb-4">
            <Star className="h-5 w-5 fill-yellow-400 text-yellow-400" />
            <span className="font-semibold text-md sm:text-lg">
              {(computedTotalSafetyScore / 20).toFixed(1)} / 5
            </span>
            <span className="text-gray-500 text-sm">
              ({computedTotalSafetyScore}%)
            </span>
          </div>
          <div className="space-y-3">
            {[{
              label: "AI Güven Skoru",
              score: aiScore,
              color: "bg-blue-500",
            }, {
              label: "Temiz İçerik",
              score: cleanScore,
              color: "bg-green-500",
            }, {
              label: "Sertifika Puanı",
              score: certScore,
              color: "bg-yellow-500",
            }, {
              label: "Kullanıcı Popülerliği",
              score: popularityScore,
              color: "bg-gray-400",
            }].map((item) => (
              <div key={item.label}>
                <div className="flex justify-between text-xs mb-1">
                  <span>{item.label}</span>
                  <span>{item.score}%</span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2">
                  <div
                    className={`h-2 rounded-full ${item.color}`}
                    style={{ width: `${item.score}%` }}
                  />
                </div>
              </div>
            ))}
          </div>

          <div className="mt-6">
            <div className="flex items-center justify-between mb-2">
              <span className="text-md font-semibold flex items-center gap-2">
                <Shield className="h-5 w-5 text-blue-600" />Genel Puan
              </span>
              <Badge
                className={`${getScoreBadgeColor(
                  computedTotalSafetyScore
                )} text-md px-3 py-1`}
              >
                {computedTotalSafetyScore}%
              </Badge>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2.5">
              <div
                className={`h-2.5 rounded-full bg-gradient-to-r ${getScoreGradient(
                  computedTotalSafetyScore
                )} transition-all duration-500`}
                style={{ width: `${computedTotalSafetyScore}%` }}
              />
            </div>
          </div>

          <div className="flex items-center gap-4 mt-4">
            <div className="flex items-center gap-1.5 text-green-600">
              <ThumbsUp className="h-4 w-4" />
              <span className="font-semibold text-sm">{product.likes}</span>
              <span className="text-xs text-gray-600">Beğeni</span>
            </div>
            <div className="flex items-center gap-1.5 text-red-600">
              <ThumbsDown className="h-4 w-4" />
              <span className="font-semibold text-sm">{product.dislikes}</span>
              <span className="text-xs text-gray-600">Beğenmeme</span>
            </div>
          </div>

          {(icerikList.length > 0 || icerikBilgisiLines.length > 0 || usageLines.length > 0) && (
            <Card className="mt-6 text-left">
              <CardContent className="p-3 space-y-2">
                {(icerikList.length > 0 || icerikBilgisiLines.length > 0) && (
                  <div>
                    <h5 className="font-semibold text-sm mb-1">İçerik Bilgisi</h5>
                    <ul className="list-disc list-inside text-xs text-gray-700 space-y-0.5">
                      {icerikList.map((item, i) => (
                        <li key={i}>{item}</li>
                      ))}
                      {icerikBilgisiLines.map((item, i) => (
                        <li key={i}>{item}</li>
                      ))}
                    </ul>
                  </div>
                )}
                {usageLines.length > 0 && (
                  <div className="mt-2">
                    <h5 className="font-semibold text-sm mb-1">Kullanım Önerisi</h5>
                    <ul className="list-disc list-inside text-xs text-gray-700 space-y-0.5">
                      {usageLines.map((item, i) => (
                        <li key={i}>{item}</li>
                      ))}
                    </ul>
                  </div>
                )}
              </CardContent>
            </Card>
          )}
        </div>
      </div>

      {product.reviews && product.reviews.length > 0 && (
        <div className="mt-6 border-t pt-5">
          <h2 className="font-semibold text-md sm:text-lg text-blue-700 mb-4">
            Kullanıcı Yorumları
          </h2>
          <div className="flex flex-col sm:flex-row gap-4 mb-4">
            <div className="flex-1">
              <Label htmlFor="rating-filter" className="text-sm font-medium">
                Yıldıza Göre
              </Label>
              <Select onValueChange={setSelectedRating} value={selectedRating}>
                <SelectTrigger id="rating-filter" className="w-full mt-1">
                  <SelectValue placeholder="Tümü" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Tüm Yıldızlar</SelectItem>
                  {[5, 4, 3, 2, 1].map((star) => (
                    <SelectItem key={star} value={String(star)}>
                      {star} Yıldız
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
            </div>
            <div className="flex-1">
              <Label htmlFor="date-filter" className="text-sm font-medium">
                Tarihe Göre
              </Label>
              <Select onValueChange={setSelectedDateFilter} value={selectedDateFilter}>
                <SelectTrigger id="date-filter" className="w-full mt-1">
                  <SelectValue placeholder="Tümü" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Tüm Zamanlar</SelectItem>
                  <SelectItem value="last_month">Son 1 Ay</SelectItem>
                  <SelectItem value="last_6_months">Son 6 Ay</SelectItem>
                  <SelectItem value="last_year">Son 1 Yıl</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
          <div className="space-y-3 max-h-[400px] overflow-y-auto pr-2 scrollbar-thin scrollbar-thumb-gray-400 scrollbar-track-gray-100">
            {filteredReviews.length === 0 ? (
              <p className="text-center text-gray-500 py-8">
                Seçilen filtrelere uygun yorum bulunamadı.
              </p>
            ) : (
              filteredReviews.map((review: any, index: number) => (
                <Card
                  key={review.id || index}
                  className="shadow-sm bg-gray-50/50"
                >
                  <CardContent className="p-3">
                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-2 gap-2">
                      <p className="font-semibold text-sm">{review.username}</p>
                      <div className="flex items-center gap-1">
                        {[...Array(review.rating)].map((_, i) => (
                          <Star
                            key={i}
                            className="h-4 w-4 fill-yellow-400 text-yellow-400"
                          />
                        ))}
                        {[...Array(5 - review.rating)].map((_, i) => (
                          <Star key={i} className="h-4 w-4 text-gray-300" />
                        ))}
                      </div>
                    </div>
                    <p className="text-xs text-gray-500 mb-2" suppressHydrationWarning>
                      {review.date && !isNaN(new Date(review.date).getTime())
                        ? new Date(review.date).toLocaleDateString()
                        : ""}
                    </p>
                    <p className="text-sm text-gray-700 leading-relaxed">
                      {review.review}
                    </p>
                  </CardContent>
                </Card>
              ))
            )}
          </div>
        </div>
      )}
    </div>
  );
}